<?php
// license-server.php - simple standalone license validation API (per-domain binding)
require __DIR__ . '/config.php';
header('Content-Type: application/json; charset=utf-8');

$resp = ['valid' => false, 'message' => 'Invalid request'];

// Accept GET params: key, domain, secret (optional)
$key = isset($_GET['key']) ? trim($_GET['key']) : '';
$domain = isset($_GET['domain']) ? strtolower(trim($_GET['domain'])) : '';
$provided_secret = isset($_GET['secret']) ? trim($_GET['secret']) : '';

// Optional API secret check
if (defined('CSA_API_SECRET') && CSA_API_SECRET !== '' ) {
    if (empty($provided_secret) || !hash_equals(CSA_API_SECRET, $provided_secret)) {
        http_response_code(403);
        $resp['message'] = 'Invalid API secret';
        echo json_encode($resp);
        exit;
    }
}

if (empty($key)) {
    http_response_code(400);
    $resp['message'] = 'Missing license key (param: key)';
    echo json_encode($resp);
    exit;
}

// connect to DB
$mysqli = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($mysqli->connect_errno) {
    http_response_code(500);
    $resp['message'] = 'Database connection failed';
    echo json_encode($resp);
    exit;
}

// lookup license
$stmt = $mysqli->prepare('SELECT id, license_key, bound_domain, active, created, expires FROM csa_licenses WHERE license_key = ? LIMIT 1');
$stmt->bind_param('s', $key);
$stmt->execute();
$result = $stmt->get_result();
$row = $result->fetch_assoc();
if (!$row) {
    http_response_code(404);
    $resp['message'] = 'License not found';
    echo json_encode($resp);
    exit;
}

// check active
if (!$row['active']) {
    $resp['message'] = 'License revoked';
    echo json_encode($resp);
    exit;
}

// check expiry
if (!empty($row['expires']) && strtotime($row['expires']) < time()) {
    $resp['message'] = 'License expired';
    echo json_encode($resp);
    exit;
}

// Domain binding logic
if (empty($row['bound_domain'])) {
    // if domain param provided, bind it
    if (!empty($domain)) {
        $bind_stmt = $mysqli->prepare('UPDATE csa_licenses SET bound_domain = ? WHERE id = ?');
        $bind_stmt->bind_param('si', $domain, $row['id']);
        $bind_stmt->execute();
        $row['bound_domain'] = $domain;
    }
}
// if license is bound, ensure domain matches (if domain param provided)
if (!empty($row['bound_domain'])) {
    if (!empty($domain) && strtolower($row['bound_domain']) !== $domain) {
        $resp['message'] = 'License bound to a different domain';
        echo json_encode($resp);
        exit;
    }
    // If domain param missing but license bound, still allow validation (best-effort)
}

$resp = [
    'valid' => true,
    'message' => 'License active',
    'license' => $row['license_key'],
    'bound_domain' => $row['bound_domain'],
    'expires' => $row['expires'],
    'created' => $row['created']
];

echo json_encode($resp);
exit;
?>